/*----------------------------------------------------------------------------
 *  Organization:  Copyright © 2021 NST, LTD. all rights reserved.
 *----------------------------------------------------------------------------*/

#ifndef MO_SDK_C_API_H_INCLUDED
    #define MO_SDK_C_API_H_INCLUDED

    #include "mo_stereo_camera_driver_macro_define.h"
    #include "mo_stereo_camera_driver_type_define.h"

    /**< function define area */
    #ifdef __cplusplus
    extern "C" {
    #endif

    /** \brief Get current SDK version
      *
      * \param no
      * \return Version
      *
      */
    DLL_API_C_DECL const char*
    moGetSdkVersion();

    /** \brief Open UVC camera by device path
      *
      * \param pcPath - the specified path string
      *        e.g."/dev/video0"
      * \param phCameraHandle - the handle of camera
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - didn't open any camera
      */
    DLL_API_C_DECL int32_t
    moOpenUVCCameraByPath(IN  const char*            const pcPath,
                          OUT      MO_CAMERA_HANDLE* const phCameraHandle);

    /** \brief Open UVC camera by video device index
      *
      * \param u8Number - The index value in /dev/video*
      *                    e.g. 0 represents /dev/video0
      * \param phCameraHandle - the handle of camera
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - didn't open any camera
      */
    DLL_API_C_DECL int32_t
    moOpenUVCCameraByNumber(IN  const uint8_t                 u8Number,
                            OUT       MO_CAMERA_HANDLE* const phCameraHandle);

    /** \brief Close specified camera with camera handle
      *
      * \param phCameraHandle - the handle of camera
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      */
    DLL_API_C_DECL int32_t
    moCloseCamera(IN MO_CAMERA_HANDLE* const phCameraHandle);

    /** \brief Suspend video streaming of specified camera
     *
     * \param hCameraHandle - The handle of camera
     * \return 0 - success, else - failure
     *        -1 - parameter is invalid
     *        -2 - command fails unexpectedly
     */
    DLL_API_C_DECL int32_t
    moSuspendCamera(IN const MO_CAMERA_HANDLE hCameraHandle);

    /** \brief Resume video streaming of specified camera
      *
      * \param hCameraHandle - The handle of camera
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moResumeCamera(IN const MO_CAMERA_HANDLE hCameraHandle);

    /** \brief Get the BxF(baseline * focus) and baseline of camera
      *
      * \param hCameraHandle - the handle of camera
      * \param pfBxf - Output value of camera baseline * focus
      *        Focus = pfBxf / pfBase
      * \param pfBase - Output value of camera baseline
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetBxfAndBase(IN const MO_CAMERA_HANDLE hCameraHandle,
                    OUT      float* const     pfBxf,
                    OUT      float* const     pfBase);

    /** \brief Get current frame data
      *        Note:
      *        Data buffer is circled and handled by SDK
      *        And data in the buffer may be refreshed after certain period
      *
      * \param hCameraHandle - the handle of camera
      * \param pu64ImageFrameNum - output frame ID
      * \param ppu8FrameBuffer - output pointer of the data buffer
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - didn't get current frame
      */
    DLL_API_C_DECL int32_t
    moGetCurrentFrame(IN  const MO_CAMERA_HANDLE hCameraHandle,
                      OUT       uint64_t* const  pu64ImageFrameNum,
                      OUT       uint8_t** const  ppu8FrameBuffer);

    /** \brief Get a data pointer to a safe buffer of current frame
      *        Note:
      *        Frame data will be copied from circled data buffer.
      *        Data will not be refreshed until another call of moGetSafeCurrentFrame
      *
      * \param hCameraHandle - the handle of camera
      * \param pu64ImageFrameNum - output frame ID
      * \param ppu8FrameBuffer - output pointer of the data buffer
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - didn't get current frame
      */
    DLL_API_C_DECL int32_t
    moGetSafeCurrentFrame(IN  const MO_CAMERA_HANDLE hCameraHandle,
                          OUT       uint64_t* const  pu64ImageFrameNum,
                          OUT       uint8_t** const  ppu8FrameBuffer);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *         to get disparity and YUV_I420 data(valid in RGBD video mode)
      *
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu16DisparityData - Output disparity data(size: Width * Height * 2 bytes)
      *                             corresponding to right camera image
      * \param ppu8YUVI420Img -Output image(YUV_I420) data(size: Width * Height * 1.5 bytes)
      *                        in the type of YUV_I420 of right camera image
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRGBDImage(IN  const MO_CAMERA_HANDLE hCameraHandle,
                   IN  const uint8_t*   const pu8FrameBuffer,
                   OUT       uint16_t** const ppu16DisparityData,
                   OUT       uint8_t**  const ppu8YUVI420Img);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get disparity data(valid in RGBD video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer -  Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu16DisparityData - Output disparity data(: Width * Height * 2 bytes)
      *                             corresponding to right camera image
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRGBDDisparityData(IN  const MO_CAMERA_HANDLE hCameraHandle,
                           IN  const uint8_t*   const pu8FrameBuffer,
                           OUT       uint16_t** const ppu16DisparityData);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get YUV_I420 data(valid in RGBD video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8YUVI420Img - Output image(YUV_I420) data(size: Width * Height * 1.5 bytes)
      *                         in the type of YUV_I420 of right camera
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRGBDYUVI420Image(IN  const MO_CAMERA_HANDLE hCameraHandle,
                          IN  const uint8_t*  const  pu8FrameBuffer,
                          OUT       uint8_t** const  ppu8YUVI420Img);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get left and right raw image data (valid in RAW video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8LeftBayerImg - output original left bayer image data (size: Width * Height * 2 bytes)
      * \param ppu8RightBayerImg - output original right bayer image data (size: Width * Height * 2 bytes)
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRawImage(IN  const MO_CAMERA_HANDLE hCameraHandle,
                  IN  const uint8_t*  const  pu8FrameBuffer,
                  OUT       uint8_t** const  ppu8LeftBayerImg,
                  OUT       uint8_t** const  ppu8RightBayerImg);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get left raw image data (valid in RAW video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8LeftBayerImg - output original left bayer image data (size: Width * Height * 2 bytes)
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRawLeftBayerImage(IN  const MO_CAMERA_HANDLE hCameraHandle,
                           IN  const uint8_t*  const  pu8FrameBuffer,
                           OUT       uint8_t** const  ppu8LeftBayerImg);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *         to get right raw image data (valid in RAW video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8RightBayerImg - output original right bayer image data (size: Width * Height * 2 bytes)
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRawRightBayerImage(IN  const MO_CAMERA_HANDLE hCameraHandle,
                            IN  const uint8_t*  const  pu8FrameBuffer,
                            OUT       uint8_t** const  ppu8RightBayerImg);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get left and right rectified image data (valid in RECTIFIED video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8LeftGrayImg - output left gray rectified image data(Size: Width * Height * 1 bytes)
      * \param ppu8RightYUVI420Img - output right YUV_I420 rectified image data(Size: Width * Height * 1.5 bytes)
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRectifiedImage(IN  const MO_CAMERA_HANDLE hCameraHandle,
                        IN  const uint8_t*  const  pu8FrameBuffer,
                        OUT       uint8_t** const  ppu8LeftGrayImg,
                        OUT       uint8_t** const  ppu8RightYUVI420Img);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get left rectified image data (valid in RECTIFIED video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8LeftGrayImg - output left gray rectified image data(Size: Width * Height * 1 bytes)
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRectifiedLeftGrayImage(IN  const MO_CAMERA_HANDLE hCameraHandle,
                                IN  const uint8_t*  const  pu8FrameBuffer,
                                OUT       uint8_t** const  ppu8LeftGrayImg);

    /** \brief Parse frame data(obtained by moGetCurrentFrame or moGetSafeCurrentFrame)
      *        to get right rectified image data (valid in RECTIFIED video mode)
      *
      * \param hCameraHandle - the handle of camera
      * \param pu8FrameBuffer - Pointer of video frame data buffer(obtained from moGetCurrentFrame/moGetSafeCurrentFrame)
      * \param ppu8RightYUVI420Img - output right YUV_I420 rectified image data(Size: Width * Height * 1.5 bytes)
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetRectifiedRightYUVI420Image(IN  const MO_CAMERA_HANDLE hCameraHandle,
                                    IN  const uint8_t*  const  pu8FrameBuffer,
                                    OUT       uint8_t** const  ppu8RightYUVI420Img);

    /** \brief Set video mode
      *
      * \param hCameraHandle - the handle of camera
      * \param eVideoMode - video mode to be set
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moSetVideoMode(IN const MO_CAMERA_HANDLE hCameraHandle,
                   IN const mo_video_mode    eVideoMode);

    /** \brief Get video mode
      *
      * \param hCameraHandle - the handle of camera
      * \param peVideoMode - output video mode
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      */
    DLL_API_C_DECL int32_t
    moGetVideoMode(IN  const MO_CAMERA_HANDLE       hCameraHandle,
                   OUT       mo_video_mode*   const peVideoMode);

    /** \brief Get video frame param
      *
      * \param hCameraHandle - the handle of camera
      * \param pstVideoFrameParam - output current video frame parameter
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetVideoFrameParam(IN const MO_CAMERA_HANDLE            hCameraHandle,
                         OUT      mo_video_frame_param* const pstVideoFrameParam);

    /** \brief Get real-time video frames per second
      *
      * \param hCameraHandle - the handle of camera
      * \param pd8FPS - the real-time video frames per second
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - Failed to get FPS(usually caused by insufficient frame counts during initial period)
      *
      */
    DLL_API_C_DECL int32_t
    moGetRealTimeFPS(IN const MO_CAMERA_HANDLE hCameraHandle,
                     OUT      double* const    pd8FPS);

    /** \brief Set fill light type
      *
      * \param hCameraHandle - the handle of camera
      * \param eFilllightType - desired fill light type
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moSetFilllightType(IN const MO_CAMERA_HANDLE hCameraHandle,
                       IN const mo_filllight_type eFilllightType);

    /** \brief Get fill light type
      *
      * \param hCameraHandle - the handle of camera
      * \param peFilllightType - output current fill light type
      * \return 0 - success, else - failure
      *        -1 - parameter is invalid
      *        -2 - command fails unexpectedly
      */
    DLL_API_C_DECL int32_t
    moGetFilllightType(IN const MO_CAMERA_HANDLE hCameraHandle,
                       OUT       mo_filllight_type* const peFilllightType);

    #ifdef __cplusplus
    }
    #endif

#endif // MO_SDK_C_API_H_INCLUDED
